#include "FriendsListSony.h"
#include "MessagePipe.h"
#include "SignInSony.h"

using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{
	CachedFriendsList gFriendsList;

	DO_EXPORT( bool, PrxFriendsListIsBusy ) ()
	{
		return gFriendsList.IsBusy();
	}

	DO_EXPORT( bool, PrxFriendsGetLastError) (ResultCode* result)
	{
		return gFriendsList.GetLastError(result);
	}

	DO_EXPORT( ErrorCode, PrxRefreshFriendsList ) ()
	{
		return gFriendsList.Refresh();
	}

	DO_EXPORT( void, PrxLockFriendsList ) ()
	{
		gFriendsList.Lock();
	}

	DO_EXPORT( void, PrxUnlockFriendsList ) ()
	{
		gFriendsList.Unlock();
	}

	DO_EXPORT( int, PrxGetFriendCount ) ()
	{
		return gFriendsList.GetFriendCount();
	}

	DO_EXPORT( ErrorCode, PrxGetFriend ) (int index, UnityPlugin::Friend* frnd)
	{
		return gFriendsList.GetFriend(index, frnd);
	}

	CachedFriendsList::CachedFriendsList()
		: m_Busy(false)
		, m_LastResult("FriendsList")
	{
	}

	bool CachedFriendsList::IsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_Busy;
	}

	void CachedFriendsList::ClearList()
	{
		for(unsigned int i=0; i<m_Friends.size(); i++)
		{
			delete m_Friends[i];
		}
		m_Friends.clear();
	}

	void CachedFriendsList::Lock()
	{
		m_Lock.Lock();
	}

	void CachedFriendsList::Unlock()
	{
		m_Lock.Unlock();
	}

	int CachedFriendsList::GetFriendCount() const
	{
		return (int)m_Friends.size();
	}

	bool CachedFriendsList::ProcessEvent(const Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;

		switch(event.event)
		{
		case Event::serviceError:
			m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
			UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_FriendsListError);
			m_FutureFriendList.reset();
			m_Busy = false;
			return true;

		case Event::friendsGotFriendsList:

			if(m_FutureFriendList.hasError())
			{
				m_LastResult.SetResultSCE(m_FutureFriendList.getError(), true, __FUNCTION__, __LINE__);
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_FriendsListError);
				m_FutureFriendList.reset();
				m_Busy = false;
				return true;
			}
			else
			{
				ClearList();

				if(m_FutureFriendList.hasResult() == false)
				{
					m_FutureFriendList.reset();
					m_Busy = false;
					return true;
				}

				FriendsList::const_iterator iter ; 
				int i = 1 ; 
				for (iter = m_FutureFriendList.get()->begin(); iter != m_FutureFriendList.get()->end() ; ++iter,i++)
				{
					CachedFriend* item = new CachedFriend;
					item->NpId = iter->npid;
					item->presence = iter->presence;
					item->presenceTitle = item->presence.title;
					item->presenceStatus = item->presence.status;
					item->state = item->presence.state;
					item->status = (unsigned char*)malloc(SCE_NP_BASIC_PRESENCE_EXTENDED_STATUS_SIZE_MAX);
					memcpy( item->status, item->presence.status, SCE_NP_BASIC_PRESENCE_EXTENDED_STATUS_SIZE_MAX);
					item->comment = (unsigned char*)malloc(SCE_NP_BASIC_PRESENCE_COMMENT_SIZE_MAX);
					memcpy( item->comment, item->presence.comment, SCE_NP_BASIC_PRESENCE_COMMENT_SIZE_MAX);
					item->presenceData = (unsigned char*)malloc(SCE_NP_BASIC_MAX_PRESENCE_SIZE);
					memcpy(item->presenceData, item->presence.data, SCE_NP_BASIC_MAX_PRESENCE_SIZE);
					item->onlineID = item->NpId.handle.data;
					item->npIdData = (unsigned char*)malloc(sizeof(SceNpId));
					memcpy(item->npIdData, &item->NpId, sizeof(SceNpId));
					m_Friends.push_back(item);
				}

				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_GotFriendsList);
			}
			m_Busy = false;
			handled = true;
			break;

		case Event::friendsNoFriends:
			ClearList();
			UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_GotFriendsList);
			m_Busy = false;
			handled = true;
			break;

		case Event::friendsListUpdated:	// List has been updated, i.e. when someones online state or presence changes.
			UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_FriendsListUpdated);
			handled = true;
			break;

		case Event::friendsListReady:	// An event generated when the user's friends list has been populated.
			// Do nothing.
			handled = true;
			break;

		case Event::friendsPresenceUpdated:
			UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolkit_FriendsPresenceUpdated);
			handled = true;
			break;
		case Event::friendsGuiExited:
		case Event::blockListReady:
		case Event::blockListUpdated:
			UnityPlugin::Messages::Log("TODO: Handle all friends list events: event=%d\n", event.event);
			handled = true;
			break;
		default:
			break;
		}

		return handled;
	}

	ErrorCode CachedFriendsList::Refresh()
	{
		SimpleLock::AutoLock lock(m_Lock);
		int ret;

		if(!gSignedInState.IsSignedIn())
		{
			return m_LastResult.SetResult(NP_ERR_NOT_SIGNED_IN, true, __FUNCTION__, __LINE__);
		}

		if(m_FutureFriendList.isBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();
		m_FutureFriendList.reset();

		ret = Friends::Interface::getFriendslist(&m_FutureFriendList, true);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedFriendsList::GetFriend(unsigned int index, Friend* frnd)
	{
		if(index >= m_Friends.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		const CachedFriend* cached = m_Friends[index];
		frnd->npPresenceTitle = cached->presenceTitle.c_str();
		frnd->npPresenceStatus = cached->presenceStatus.c_str();
		frnd->npID = cached->npIdData;
		frnd->npIDSize = sizeof(SceNpId);
		frnd->npOnlineID = cached->onlineID.c_str();
		frnd->npOnlineStatus = cached->state;	//	TODO : extract cached->presence.status
		frnd->npPresenceSdkVersion = 0;
		frnd->npPresenceType = 0;
		frnd->npPresenceDataSize = 0;
		frnd->npComment = (const char*)cached->comment;
		frnd->npPresenceTitle = cached->presenceTitle.c_str();
		frnd->npPresenceStatus = cached->presenceStatus.c_str();
		frnd->npPresenceData = cached->presenceData;

		return m_LastResult.GetResult();
	}
}; //namespace UnityPlugin
